/* Copyright (c) 2004 Brian Dam Pedersen
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <Python.h>
#include <Numeric/arrayobject.h>


typedef long* I_t;
typedef double* D_t;


extern void ql0001_(I_t M, I_t ME, I_t MMAX, I_t N, I_t NMAX, I_t MNN, 
		    D_t C, D_t D, D_t A,D_t B, D_t XL, D_t XU,
		    D_t X, D_t U, I_t IOUT, I_t IFAIL,I_t IPRINT, D_t WAR,
		    I_t LWAR, I_t IWAR, I_t LIWAR);

// x,lgm=qp(C,d,A,b [,eq_c][,xl][,xu], lagrange=1)
// x=qp(C,d,A,b [,eq_c][,xl][,xu])
static PyObject* qp(PyObject *self, PyObject *args, PyObject *kwargs)
{
  // Placeholders for input and output parameters
  PyArrayObject *b, *d, *xl, *xu; 
  PyArrayObject *A, *C, *bph, *dph, *xlph, *xuph, *x, *lgm;
  int lagrange, eq_c, nvars;
  // Worker arrays, we need to convert A and C to column-major
  // and we need WAR and IWAR to be here also
  double *Ac, *Cc, *WAR, *dptr;
  long *IWAR;
  // Control integers for the routine
  long M, ME, MMAX, N, NMAX, MNN, IOUT, IFAIL, IPRINT, LWAR, LIWAR,r,c;
  int itmp;
  // Return value
  PyObject *retval;
  // parameter names
  static char *kwlist[]={"C","d","A","b","eq_c","xl","xu","lagrange","output",NULL};
  char ctmp[80];

  // Initialize inputs to NULL so that we can see who actually came by :)
  A=NULL;
  b=NULL;
  C=NULL;
  d=NULL;
  xl=NULL;
  xu=NULL;
  lagrange = 0;
  eq_c = 0;
  itmp = 0; 
  // Parse input
  if (!PyArg_ParseTupleAndKeywords(args,kwargs,"O!O!O!O!|iO!O!ii",kwlist,
				   &PyArray_Type,&C,
				   &PyArray_Type,&d,
				   &PyArray_Type,&A,
				   &PyArray_Type,&b,
				   &eq_c,
				   &PyArray_Type,&xl, 
				   &PyArray_Type,&xu,
				   &lagrange,&itmp))
    return NULL;
  IPRINT=itmp;

  // Dimensionality check
  if (C->nd != 2){
    PyErr_SetString(PyExc_ValueError,"C must be two-dimensional");
    return NULL;
  }
  if (d->nd !=1){
    PyErr_SetString(PyExc_ValueError,"d must be one-dimensional");
    return NULL;
  }
  if (A->nd != 2){
    PyErr_SetString(PyExc_ValueError,"A must be two-dimensional");
    return NULL;
  }
  if (b->nd !=1){
    PyErr_SetString(PyExc_ValueError,"b must be one-dimensional");
    return NULL;
  }
  if (xl){
    if (xl->nd != 1){
      PyErr_SetString(PyExc_ValueError,"xl must be one-dimensional");
      return NULL;
    }
  }
  if (xu){
    if (xu->nd != 1){
      PyErr_SetString(PyExc_ValueError,"xu must be one-dimensional");
      return NULL;
    }
  }

  // Type check
  if (C->descr->type_num != PyArray_DOUBLE){
    PyErr_SetString(PyExc_TypeError,"C must be of type Float");
    return NULL;
  }
  if (d->descr->type_num != PyArray_DOUBLE){
    PyErr_SetString(PyExc_TypeError,"d must be of type Float");
    return NULL;
  }
  if (A->descr->type_num != PyArray_DOUBLE){
    PyErr_SetString(PyExc_TypeError,"A must be of type Float");
    return NULL;
  }
  if (b->descr->type_num != PyArray_DOUBLE){
    PyErr_SetString(PyExc_TypeError,"b must be of type Float");
    return NULL;
  }
  if (xl){
    if (xl->descr->type_num != PyArray_DOUBLE){
      PyErr_SetString(PyExc_TypeError,"xl must be of type Float");
      return NULL;
    }
  }
  if (xu){
    if (xu->descr->type_num != PyArray_DOUBLE){
      PyErr_SetString(PyExc_TypeError,"xu must be of type Float");
      return NULL;
    }
  }

  // Check consistency 
  if (A->dimensions[0] != b->dimensions[0]){
    PyErr_SetString(PyExc_ValueError,"Number of rows in A must match number of elements in b");
    return NULL;
  }
  if (C->dimensions[0] != C->dimensions[1]){
    PyErr_SetString(PyExc_ValueError,"C must be square");
    return NULL;
  }
  if (C->dimensions[0] != d->dimensions[0]){
    PyErr_SetString(PyExc_ValueError,"Number of rows in C must match number of elements in d");
    return NULL;
  }
  if (C->dimensions[1] != A->dimensions[1]){
    PyErr_SetString(PyExc_ValueError,"C and A must have the same number of columns");
    return NULL;
  }

  if (xl)
    if (C->dimensions[1] != xl->dimensions[0]){
      PyErr_SetString(PyExc_ValueError,"Number fof columns in C must match number of elements in xl");
      return NULL;
    }
  if (xu)
    if (C->dimensions[1] != xu->dimensions[0]){
      PyErr_SetString(PyExc_ValueError,"Number fof columns in C must match number of elements in xu");
      return NULL;
    }



  // align data and possibly allocate and init xl and xu
  nvars=C->dimensions[1];
  bph=(PyArrayObject*)
    PyArray_ContiguousFromObject((PyObject*)b,PyArray_DOUBLE,1,1);
  dph=(PyArrayObject*)
    PyArray_ContiguousFromObject((PyObject*)d,PyArray_DOUBLE,1,1);

  if (xl){
    xlph=(PyArrayObject*)
      PyArray_ContiguousFromObject((PyObject*)xl, PyArray_DOUBLE, 1,1);
  }else{
    xlph=(PyArrayObject*)PyArray_FromDims(1,&nvars,PyArray_DOUBLE);
    dptr=(double*)xlph->data;
    for (r=0; r < nvars; r++, dptr++) *dptr=(double)(-1e16);
  }
  if (xu){
    xuph=(PyArrayObject*)
      PyArray_ContiguousFromObject((PyObject*)xu, PyArray_DOUBLE, 1,1);
  }else{
    xuph=(PyArrayObject*)PyArray_FromDims(1,&nvars,PyArray_DOUBLE);
    dptr=(double*)xuph->data;
    for (r=0; r < nvars; r++, dptr++) *dptr=(double)(1e16);
  }
  // Initialize control variables
  M=A->dimensions[0]; 
  ME=eq_c; 
  MMAX=M+1; 
  N=C->dimensions[1]; 
  NMAX=N; 
  MNN=M+N+N; 
  IOUT=6; 
  LWAR=3*NMAX*NMAX/2 + 10*NMAX + 2*MMAX+2; 
  LIWAR=N;
  

  // Allocate some extra data
  x=(PyArrayObject*)PyArray_FromDims(1,&nvars,PyArray_DOUBLE);
  itmp=MNN;
  lgm=(PyArrayObject*)PyArray_FromDims(1,&itmp,PyArray_DOUBLE);
  Cc=(double*)alloca(sizeof(double)*N*N);
  Ac=(double*)alloca(sizeof(double)*MMAX*MMAX);
  WAR=(double*)alloca(sizeof(double)*LWAR);
  IWAR=(long*)alloca(sizeof(long)*LIWAR);
 
  IWAR[0]=1;

  // copy C and A to column-major
  for (c=0 ; c < C->dimensions[1] ; c++)
    for (r=0 ; r < C->dimensions[0] ; r++)
      Cc[r+NMAX*c]=*(double*)(C->data + r*C->strides[0] + c*C->strides[1]);
  for (c=0 ; c < A->dimensions[1] ; c++)
    for (r=0 ; r < A->dimensions[0] ; r++)
      Ac[r+MMAX*c]=*(double*)(A->data + r*A->strides[0] + c*A->strides[1]);

  // Perform QP
  ql0001_(&M, &ME, &MMAX, &N, &NMAX, &MNN, 
	  Cc, (double*)dph->data, Ac ,(double*)bph->data, 
	  (double*)xlph->data , (double*)xuph->data,
	  (double*)x->data, (double*)lgm->data, &IOUT, &IFAIL,&IPRINT, WAR,
	       &LWAR, IWAR, &LIWAR);


  // Check and construct return value - default is exception
  retval=NULL;
  switch (IFAIL){
  case 0: /* success*/
    Py_INCREF(x);
    if (lagrange){
      Py_INCREF(lgm);
      retval=Py_BuildValue("(NN)",PyArray_Return(x),PyArray_Return(lgm));
    }else{
      retval=PyArray_Return(x);
    }
    break;
  case 1: /* too many iterations*/
    PyErr_SetString(PyExc_ArithmeticError,"Too many iterations");
    break;
  case 2: /* Insufficient accuracy*/
    PyErr_SetString(PyExc_ArithmeticError,"Insufficient accuracy");
    break;
  case 5: /* Length of a working array is too short*/
    PyErr_SetString(PyExc_SystemError,"Oops - bug in the wrapper, please contact author");
    break;
  default: /* Inconsistent constraints */
    itmp=IFAIL;
    sprintf(ctmp,"Inconsistent constraints (err=%d )",itmp);
    PyErr_SetString(PyExc_ValueError,ctmp );
    break;
  }

  // decref placeholders
  Py_DECREF(bph);
  Py_DECREF(dph);
  Py_DECREF(xlph);
  Py_DECREF(xuph);
  // decref outputs - if we need keeping them, they were increfed above
  Py_DECREF(x);
  Py_DECREF(lgm);
  
  // Return appropriate value(s)
  return retval;
}

static PyMethodDef qp_methods[] = 
  {
    {"qp", (PyCFunction)qp, METH_VARARGS | METH_KEYWORDS,
     "Quadratic programming routine. Syntax is:\n"
     "x=qp(C,d,A,b [,eq_c][,xl][,xu])\n"
     "x,lgm=qp(C,d,A,b [,eq_c][,xl][,xu], lagrange=1)\n"
     "\n"
     "where:\n"
     "C (array of Float length nxn) : Objective function matrix. Must be\n"
     "                                symmetric  and positive definite\n"
     "\n"
     "d (vector of Float length n)  : Constant vector of the objective function\n"
     "\n"
     "A (array of FLoat length mxn) : Contains the data matrix for the linear\n" 
     "                                constraints\n"
     "\n"
     "b (vector of Float length m)  : Contains the constant data for the linear\n"
     "                                constraints\n"
     "\n"
     "eq_c (integer)                : Number of equality constraints. These are \n"
     "                                always described by the first row(s) in A\n"
     "                                and B. Default is 0\n"
     "\n"
     "xl (vector of Float length n) : Lower bounds for the variables. Default is\n"
     "                                all elements equal to -10e16\n"
     "\n"
     "xu (vector of Float length m) : Upper bounds for the variables. Default is\n"
     "                                all elements equal to 10e16\n"
     "\n"
     "lagrange (integer)            : If non-zero the return value is a tuple\n"
     "                                containing (x,lgm). If zero only x is \n"
     "                                returned (not in a tuple). Default is 0\n"
     "\n"
     "x (vector of Float length n)  : Solution to the quadratic problem\n"
     "\n"
     "lgm (vector of Float length\n"
     "                       m+n+n) : Lagrange multipliers. The first M positions\n"
     "                                are reserved for the multipliers of the M\n"
     "                                linear constraints and the subsequent ones\n"
     "                                for the multipliers of the lower and upper\n"
     "                                bounds. On successful termination, all\n"
     "                                values of U with respect to inequalities\n"
     "                                and bounds should be greater than or equal\n"
     "                                to zero.\n"
     "\n"
     "\n"
     "QP.qp() solves the quadratic programming problem\n"
     "\n"
     "Minimize        .5*x'*C*x + d'*x\n"
     "subject to      A(J)*x  +  b(J)   =  0  ,  J=1,...,eq_c\n"
     "                A(J)*x  +  b(J)  >=  0  ,  J=eq_c+1,...,m\n"
     "                xl  <=  x  <=  xu\n"},
    {NULL,NULL,0,NULL}
  };

 
void initQP(void)
{
  Py_InitModule("QP",qp_methods);
  import_array();
}




